#include <io.h>
#include <fcntl.h>
#include <stdio.h>
#include <limits.h>
#include <string.h>

#ifndef _STDINT_H
typedef signed char    int8_t;
typedef unsigned char  uint8_t;
typedef short          int16_t;
typedef unsigned short uint16_t;
/* WIN */
#if INT_MAX == 0x7FFFFFFFL
typedef int            int32_t;
typedef unsigned int   uint32_t;
#ifndef _WIN32
#ifdef __WIN32
#define _WIN32 1
#else
#ifdef __WIN32__
#define _WIN32 1
#endif
#endif
#endif
#ifdef _WIN32
#define AM_VER "Win32"
#else
#define AM_VER "Arc32"
#endif
#endif
/* DOS */
#if INT_MAX == 0x7FFF
typedef long            int32_t;
typedef unsigned long  uint32_t;
#define AM_VER "DOS16"
#endif
#endif

/* some hacks to reduce win32 executable file size */
#ifdef _WIN32
#include "win32gcc.h"
#include "intwfile.h"
#endif

/* "QAV\x1A" */
#define kQAVSig     0x1A564151
#define kQAVVersion 0x0200
#define kMaxLayers  8

enum {
  /* begin rotatesprite compatible flags */
  kQFrameNormal       = 0,    /* no flags */
  kQFrameTranslucent  = 0x01, /* frame is translucent */
  kQFrameScale        = 0x02, /* frame is scaled to viewing window */
  kQFrameYFlip        = 0x04, /* frame is y-flipped */
  kQFrameUnclipped    = 0x08, /* frame is not clipped to umost/dmost */
  kQFrameStatus       = 0x0A, /* frame is not masked (used for status bar) */
  kQFrameCorner       = 0x10, /* frame is positioned by top left corner instead of origin */
  kQFrameTranslucentR = 0x20, /* frame is translucent, using reverse translucency table */
  kQFrameNoMask       = 0x40,
  kQFrameAllPages     = 0x80,
  /* begin internal flags */
  kQFrameXFlip        = 0x100 /* frame is x-flipped */
};

#pragma pack(push, 1)

typedef struct {
  int32_t id;
} TRIGGER_FRAME;

typedef struct {
  int32_t  id; /* >0 - sound frame exists */
  uint8_t  priority;
  uint8_t  volume;
  uint16_t soundAngle;
} SOUND_FRAME;

typedef struct {
  int32_t  id;    /* >0 - tile frame exists */
  int32_t  x, y;
  int32_t  zoom;  /* zoom in 16:16 fixed format */
  int32_t  flags; /* see enum type above */
  int8_t   shade;
  uint8_t  pal;
  uint16_t angle; /* angle in Build units (0 is straight up) */
} TILE_FRAME;

/* sizeof(FRAME) = 204 bytes */
typedef struct {
  TRIGGER_FRAME trigger;
  SOUND_FRAME   sound;
  TILE_FRAME    layer[kMaxLayers];
} FRAME;

typedef struct {
  int32_t x;
  int32_t y;
} Point;

/* sizeof(qav_head) = 36 bytes */
typedef struct {
  int32_t signature; /* char  signature[4]; */
  int16_t version;
  char    dummy[2];
  int32_t nFrames;
  int32_t ticksPerFrame; /* inverted play rate */
  int32_t duration;      /* length in ticks */ /* (nFrames * ticksPerFrame) */
  Point   origin;
  char    reserved[8];
} qav_head;

/* .SEQ */

/* "SEQ\x1A" */
#define kSEQSig       0x1A514553
#define kSEQVersion   0x0300
#define kMaxSequences 1024

/* sequence flags */
enum {
  kSeqLoop   = 1,
  kSeqRemove = 2
};

typedef struct {
  uint16_t nTile         : 12;
  uint16_t translucent   : 1;
  uint16_t translucentR  : 1;
  uint16_t blocking      : 1;
  uint16_t hitscan       : 1;
  uint16_t xrepeat       : 8;
  uint16_t yrepeat       : 8;

  int16_t  shade         : 8;
  uint16_t pal           : 5;
  uint16_t trigger       : 1; /* trigger callback */
  uint16_t smoke         : 1; /* add smoke tsprite */

/*  uint16_t reserved     : 17; */
  uint16_t shadeRelative : 1;
  uint16_t reserved      : 16;
} SEQFRAME;

typedef struct {
  uint32_t signature;     /* char  signature[4]; */
  int16_t  version;
  int16_t  nFrames;       /* sequence length */
  int16_t  ticksPerFrame; /* inverted play rate */
  int16_t  soundID;
  uint8_t  flags;
  char     pad[3];
} seq_head;

#pragma pack(pop)

int32_t StrToInt(char *s) {
int32_t result;
  result = 0;
  for (; *s; s++) {
    if ((*s >= '0') && (*s <= '9')) {
      result *= 10;
      result += (*s - '0');
    } else {
      result = 0;
      break;
    }
  }
  return(result);
}

#define MODE_MOV 0
#define MODE_ADD 1
#define MODE_SUB 2
#define MODE_BRK 3
#define ID_MIN 1L
#define ID_MAX 4095L
#define ID_QMX 6143L

int32_t CheckRange(int32_t v, int32_t d, int32_t l, int32_t h, int m) {
int32_t r;
  r = 0;
  switch (m) {
    case MODE_MOV:
      r = d;
      break;
    case MODE_ADD:
      r = v + d;
      if (r > h) { r = 0; }
      break;
    case MODE_SUB:
      r = v - d;
      if (r < l) { r = 0; }
      break;
  }
  return(r);
}

int CheckParms(int32_t v, int32_t *vn, int32_t *vx, int32_t cn, int32_t cx) {
  if (*vn < cn) {
    printf("Error: invalid min value - must be bigger or equal to %ld.\n\n", cn);
    return(0);
  }
  if (*vx > cx) {
    printf("Error: invalid max value - must be smaller or equal to %ld.\n\n", cx);
    return(0);
  }
  if (*vn >= *vx) {
    printf("Error: min value must be smaller than max value.\n\n");
    return(0);
  }
  if ((v < cn) || (v > cx)) {
    printf("Error: invalid id value - abs must be in range %ld <= value <= %ld.\n\n", cn, cx);
    return(0);
  }
  return(1);
}

int main(int argc, char *argv[]) {
qav_head qh;
seq_head sh;
FRAME    qf;
SEQFRAME sf;
char *s;
int j, k, md, dt, fl;
int32_t i, v, v_min, v_max;
  printf("Blood .SEQ/.QAV tiles id number fixer v1.0 ["AM_VER"]\n(c) -=CHE@TER=- 2017\nhttp://ctpax-cheater.losthost.org/\n\n");
  if ((argc < 3) || (argc > 5)) {
    printf(
      "Usage: seqavfix <filename.ext> [+|-]<value> [min] [max]\n"
      "Where:\n"
      "  filename.ext - .SEQ or .QAV file name to change id\n"
      "  value - replace all id tile numbers with the specified value\n"
      "  +value - add value to all existing id tile numbers\n"
      "  -value - substruct value from all existing id tile numbers\n"
      "  min - don't change id smaller than this (optional, default %ld)\n"
      "  max - don't change id bigger than this (optional, .SEQ: %ld, .QAV: %ld)\n"
      "Example (shift tile id numbers by 256):\n"
      "  seqavfix test.qav t +256\n\n",
      ID_MIN,
      ID_MAX,
      ID_QMX
    );
    return(1);
  }
  fl = open(argv[1], O_RDWR | O_BINARY);
  if (fl == -1) {
    printf("Error: can not open input file for read-write.\n\n");
    return(2);
  }
  /* get filesize */
  lseek(fl, 0, SEEK_END);
  i = tell(fl);
  /* read header as .SEQ */
  lseek(fl, 0, SEEK_SET);
  memset(&sh, 0, sizeof(sh));
  read(fl, &sh, sizeof(sh));
  /* read header as .QAV */
  lseek(fl, 0, SEEK_SET);
  memset(&qh, 0, sizeof(qh));
  read(fl, &qh, sizeof(qh));
  dt = 0;
  if (
    /* .SEQ */
    ((sh.signature == kSEQSig) && (sh.version == kSEQVersion) && (i >= (sizeof(sh) + (sh.nFrames * sizeof(sf))))) ||
    /* .QAV */
    ((qh.signature == kQAVSig) && (qh.version == kQAVVersion) && (i >= (sizeof(qh) + (qh.nFrames * sizeof(qf)))))
  ) {
    /* data type */
    dt = (qh.signature == kQAVSig);
  } else {
    close(fl);
    printf("Error: Not a valid .SEQ / .QAV file or unsupported version.\n\n");
    return(3);
  }
  /* parse commandline */
  md = MODE_MOV;
  s = argv[2];
  if ((*s == '+') || (*s == '-')) {
    md = (*s == '+') ? MODE_ADD : MODE_SUB;
    s++;
  }
  v = StrToInt(s);
  v_min = (argc > 3) ? StrToInt(argv[3]) : ID_MIN;
  v_max = (argc > 4) ? StrToInt(argv[4]) : (dt ? ID_QMX : ID_MAX);
  if (!CheckParms(v, &v_min, &v_max, ID_MIN, (dt ? ID_QMX : ID_MAX))) {
    close(fl);
    return(4);
  }
  printf("%s - ", argv[1]);
  for (k = 0; k < 2; k++) {
    lseek(fl, (dt ? sizeof(qh) : sizeof(sh)), SEEK_SET);
    /* for each frame */
    for (i = 0; i < (dt ? qh.nFrames : sh.nFrames); i++) {
      if (dt) {
        /* .QAV */
        read(fl, &qf, sizeof(qf));
        /* for each layer */
        for (j = 0; j < kMaxLayers; j++) {
          /* active layer */
          if (qf.layer[j].id >= v_min) {
            if (!k) {
              /* check */
              if (!CheckRange(qf.layer[j].id, v, v_min, v_max, md)) {
                printf(
                  "error\n\nOverflow in frame(%ld).layer(%d).id == %ld [%ld..%ld]\n\n",
                  i + 1, j + 1, qf.layer[j].id, v_min, v_max
                );
                md = MODE_BRK;
                break;
              }
            } else {
              qf.layer[j].id = CheckRange(qf.layer[j].id, v, v_min, v_max, md);
            }
          }
        }
      } else {
        /* .SEQ */
        read(fl, &sf, sizeof(sf));
        /* active layer */
        if (sf.nTile >= v_min) {
          if (!k) {
            /* check */
            if (!CheckRange(sf.nTile, v, v_min, v_max, md)) {
              printf(
                "error\n\nOverflow in frame(%ld).nTile == %d [%ld..%ld]\n\n",
                 i + 1, sf.nTile, v_min, v_max
              );
              md = MODE_BRK;
            }
          } else {
            sf.nTile = CheckRange(sf.nTile, v, v_min, v_max, md);
          }
        }
      }
      if (md == MODE_BRK) { break; }
      /* write anything only on second step */
      if (k) {
        if (dt) {
          lseek(fl, tell(fl) - sizeof(qf), SEEK_SET);
          write(fl, &qf, sizeof(qf));
        } else {
          lseek(fl, tell(fl) - sizeof(sf), SEEK_SET);
          write(fl, &sf, sizeof(sf));
        }
      }
    }
    if (md == MODE_BRK) { break; }
  }
  if (md != MODE_BRK) {
    /* truncate file (mostly for .SEQ) files */
/*    chsize(fl, tell(fl)); */
    printf("done\n\n");
  }
  close(fl);
  return(0);
}
