#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <stdint.h>

#pragma pack(push, 1)
typedef struct {
  uint8_t  IDLength;
  uint8_t  ColorMapType;
  uint8_t  ImageType;
  uint16_t CMapStart;
  uint16_t CMapLength;
  uint8_t  CMapDepth;
  uint16_t XOffset;
  uint16_t YOffset;
  uint16_t Width;
  uint16_t Height;
  uint8_t  PixelDepth;
  uint8_t  ImageDescriptor;
} tga_head;
#pragma pack(pop)

#define IMG_RLE 1
#define IMG_WXH 2
#define IMG_POS 4
#define IMG_PAL 8
#define IMG_BMP 16

char *basename(char *s) {
char *r;
  if (s) {
    for (r = s; *r; r++) {
      if ((*r == '/') || (*r == '\\')) {
        s = &r[1];
      }
    }
  }
  return(s);
}

void newext(char *name, char *ext) {
char *p;
  if (name && ext) {
    for (; (*name == '.'); name++);
    for (p = NULL; *name; name++) {
      if (*name == '.') {
        p = name;
      }
    }
    strcpy(p ? p : name, ext);
  }
}

uint32_t StrToInt(char *s) {
uint32_t r;
  r = 0;
  while (*s) {
    r *= 10;
    if ((*s >= '0') && (*s <= '9')) {
      r += (*s - '0');
      s++;
    } else {
      break;
    }
  }
  return(r);
}

uint8_t FindNearest(uint8_t *p, uint8_t r, uint8_t g, uint8_t b) {
uint32_t i, d, m;
int32_t x, y, z;
uint8_t v;
  v = 0;
  if (p) {
    /* set min distance as bigger than maximum possible */
    m = (255 * 255 * 3) + 1;
    for (i = 0; i < 256; i++) {
      /* allowed color */
      if (p[3]) {
        /* calc distance */
        x = r - p[0];
        y = g - p[1];
        z = b - p[2];
        /* it's faster without square root */
        d = (x * x) + (y * y) + (z * z);
        if (d < m) {
          m = d;
          v = i;
        }
      }
      p += 4;
    }
  }
  return(v);
}

void TGAFlip(tga_head *h, uint8_t *p) {
uint32_t i, j, k, l;
uint8_t *a, *b;
  if (h && p && ((h->ImageDescriptor & 0x30) != 0x20)) {
    l = (h->PixelDepth + 7) / 8;
    /* flip bottom-top */
    if (!(h->ImageDescriptor & 0x20)) {
      h->ImageDescriptor ^= 0x20;
      a = p;
      b = &p[(h->Height - 1) * (l * h->Width)];
      for (j = 0; j < h->Height / 2; j++) {
        for (i = 0; i < h->Width; i++) {
          for (k = 0; k < l; k++) {
            a[k] ^= b[k];
            b[k] ^= a[k];
            a[k] ^= b[k];
          }
          a += l;
          b += l;
        }
        b -= (h->Width * l * 2);
      }
    }
    /* flip right-left */
    if (h->ImageDescriptor & 0x10) {
      h->ImageDescriptor ^= 0x10;
      for (j = 0; j < h->Height; j++) {
        a = &p[j * h->Width * l];
        b = &a[(h->Width - 1) * l];
        for (i = 0; i < h->Width / 2; i++) {
          for (k = 0; k < l; k++) {
            a[k] ^= b[k];
            b[k] ^= a[k];
            a[k] ^= b[k];
          }
          a += l;
          b -= l;
        }
      }
    }
  }
}

int main(int argc, char *argv[]) {
uint8_t palold[256 * 4], palnew[256 * 3];
uint32_t i, j, sz;
uint8_t *p, *u;
tga_head th;
char s[260];
FILE *fl;
  if ((argc != 3) && (argc != 5)) {
    printf("Usage: nhcimage <new_file.tga> <old_file.tga> [xpos] [ypos]\n\n");
    return(1);
  }
  /* from file */
  fl = fopen(argv[2], "rb");
  if (!fl) {
    printf("Error: can't open old_file for read.\n\n");
    return(2);
  }
  memset(&th, 0, sizeof(th));
  fread(&th, sizeof(th), 1, fl);
  if (
    (th.ColorMapType != 1) || (th.ImageType != 1) || (th.PixelDepth != 8) ||
    (th.CMapDepth != 24) || (th.CMapStart != 0) || (th.CMapLength != 256)
  ) {
    fclose(fl);
    printf("Error: old_file must be 8 BPP uncompressed image with full 24 BPP palette.\n\n");
    return(3);
  }
  fseek(fl, sizeof(th) + th.IDLength, SEEK_SET);
  for (i = 0; i < 256; i++) {
    /* drop color used flag */
    palold[(i * 4) + 3] = 0;
    /* BGR -> RGB */
    fread(&palold[(i * 4) + 2], 1, 1, fl);
    fread(&palold[(i * 4) + 1], 1, 1, fl);
    fread(&palold[(i * 4) + 0], 1, 1, fl);
  }
  sz = th.Width * th.Height;
  p = (uint8_t *) malloc(sz);
  if (p) { fread(p, sz, 1, fl); }
  fclose(fl);
  if (!p) {
    printf("Error: can't allocate memory for old_file data.\n\n");
    return(4);
  }
  /* build list of actually used colors in original image */
  for (i = 0; i < sz; i++) { palold[(p[i] * 4) + 3] = 1; }
  free(p);
  printf("[1/3] Old image read and parsed.\n");
  /* dest file */
  fl = fopen(argv[1], "rb");
  if (!fl) {
    printf("Error: can't open new_file for read.\n\n");
    return(5);
  }
  memset(&th, 0, sizeof(th));
  fread(&th, sizeof(th), 1, fl);
  if (
    ( /* 8 BPP */
      (th.ColorMapType != 1) || (th.ImageType != 1) || (th.PixelDepth != 8) ||
      (th.CMapDepth != 24) || (th.CMapStart != 0) || (th.CMapLength != 256)
    ) && (
      /* 24 BPP */
      (th.ColorMapType != 0) || (th.ImageType != 2) || (th.PixelDepth != 24)
    )
  ) {
    fclose(fl);
    printf("Error: new_file must be 24 BPP image or 8 BPP with full palette.\n\n");
    return(6);
  }
  /* check coordinates overflow */
  if (argc == 5) {
    i = StrToInt(argv[3]);
    j = StrToInt(argv[4]);
    if (((i + th.Width) > 640) || ((j + th.Height) > 480)) {
      fclose(fl);
      printf(
        "Error: x + w > 640 (%u + %u = %u) or y + h > 480 (%u + %u = %u).\n\n",
        i, th.Width, i + th.Width, j, th.Height, j + th.Height
      );
      return(7);
    }
  }
  fseek(fl, sizeof(th) + th.IDLength, SEEK_SET);
  /* 8 BPP - read palette */
  if (th.PixelDepth == 8) { fread(palnew, sizeof(palnew), 1, fl); }
  sz = th.Width * th.Height * (th.PixelDepth / 8);
  p = (uint8_t *) malloc(sz);
  if (p) {
    memset(p, 0, sz);
    fread(p, sz, 1, fl);
  }
  fclose(fl);
  if (!p) {
    printf("Error: can't allocate memory for new_file data.\n\n");
    return(8);
  }
  TGAFlip(&th, p);
  /* 8 BPP */
  if (th.PixelDepth == 8) {
    /* convert palette */
    for (i = 0; i < 256; i++) {
      palnew[i] = FindNearest(palold, palnew[(i * 3) + 2], palnew[(i * 3) + 1], palnew[(i * 3) + 0]);
    }
    /* convert image colors */
    for (i = 0; i < sz; i++) { p[i] = palnew[p[i]]; }
  } else {
    /* 24 BPP */
    sz = th.Width * th.Height;
    u = (uint8_t *) malloc(sz);
    if (!u) {
      free(p);
      printf("Error: can't allocate memory to convert 24 BPP image data.\n\n");
      return(9);
    }
    /* convert image colors */
    for (i = 0; i < sz; i++) {
      u[i] = FindNearest(palold, p[(i * 3) + 2], p[(i * 3) + 1], p[(i * 3) + 0]);
    }
    free(p);
    p = u;
  }
  /* update file */
  printf("[2/3] New image read and parsed.\n");
  /* create game image format */
  strcpy(s, basename(argv[1]));
  newext(s, ".N02");
  fl = fopen(s, "wb");
  if (!fl) {
    free(p);
    printf("Error: can't create output file \"%s\".\n\n", s);
    return(11);
  }
  i = IMG_WXH | IMG_BMP;
  i |= (argc == 5) ? IMG_POS : IMG_PAL;
  fwrite(&i, 2, 1, fl);
  /* image has size */
  if (i & IMG_WXH) {
    fwrite(&th.Width, 2, 1, fl);
    fwrite(&th.Height, 2, 1, fl);
  }
  /* image has position */
  if (i & IMG_POS) {
    j = StrToInt(argv[3]);
    fwrite(&j, 2, 1, fl);
    j = StrToInt(argv[4]);
    fwrite(&j, 2, 1, fl);
  }
  /* image has palette */
  if (i & IMG_PAL) {
    for (j = 0; j < 256; j++) {
      palold[(j * 4) + 3] = 0;
      fwrite(&palold[j * 4], 4, 1, fl);
    }
  }
  /* image has bitmap */
  if (i & IMG_BMP) {
    /* align padding bytes */
    i = (4 - (th.Width & 3)) & 3;
    sz = 0;
    for (j = 0; j < th.Height; j++) {
      /* write row of pixels */
      fwrite(&p[j * th.Width], th.Width, 1, fl);
      /* write align padding bytes */
      fwrite(&sz, i, 1, fl);
    }
  }
  fclose(fl);
  free(p);
  printf("[3/3] New image saved as \"%s\".\ndone\n\n", s);
  return(0);
}
